

#include "elf-image-cache.h"

/*
 * The following are internals of libbacktrace. It would be fantastic if we
 * did not need to do this but could use official interfaces of
 * libbacktrace only.
 */

#include <libbacktrace/internal.h>

static const char * const dwarf_section_names[DEBUG_MAX] =
{
  ".debug_info",
  ".debug_line",
  ".debug_abbrev",
  ".debug_ranges",
  ".debug_str",
  ".debug_addr",
  ".debug_str_offsets",
  ".debug_line_str",
  ".debug_rnglists"
};

static void setup_dwarf_state(struct backtrace_state *state,
                              struct elf_image *ei,
                              backtrace_error_callback error_cb, void *data)
{
  if (state->fileline_fn)
    return;

  struct dwarf_sections dwarf_sections;
  memset (&dwarf_sections, 0, sizeof(dwarf_sections));
  for (int i = 0; i < DEBUG_MAX; i++)
    {
      Elf_W(Shdr) *shdr = elf_w(find_section)(ei, dwarf_section_names[i]);
      if (shdr)
        {
          dwarf_sections.data[i] = ei->image + shdr->sh_offset;
          dwarf_sections.size[i] = shdr->sh_size;
        }
    }

  Elf_W(Ehdr) *ehdr = ei->image;
  struct libbacktrace_base_address baddr = { .m = 0 };
  backtrace_dwarf_add (state, baddr, &dwarf_sections,
                       ehdr->e_ident[EI_DATA] == ELFDATA2MSB,
                       NULL, error_cb, data, &state->fileline_fn, NULL);
}

typedef struct elf_image_cache_item
{
  elf_image_handle_t elf;
  struct backtrace_state state;
  struct elf_image_cache_item *next;
} elf_image_cache_item_t;


static elf_image_cache_item_t *elf_image_cache = NULL;

elf_image_handle_t *get_elf_image(char const *path)
{
  elf_image_cache_item_t *item = elf_image_cache;
  while (item)
    {
      if (!strcmp(item->elf.path, path))
        return &item->elf;
      item = item->next;
    }

  item = malloc(sizeof(elf_image_cache_item_t));
  if (!item)
    return NULL;

  memset(item, 0, sizeof(elf_image_cache_item_t));

  strncpy(item->elf.path, path, sizeof(item->elf.path));
  item->elf.path[sizeof(item->elf.path) - 1] = '\0';

  if (elf_map_image(&item->elf.ei, item->elf.path) < 0)
    {
      free(item);
      return NULL;
    }

  item->next = elf_image_cache;
  elf_image_cache = item;
  return &item->elf;
}

struct backtrace_state *get_backtrace_state(elf_image_handle_t *elf,
                                            backtrace_error_callback error_cb,
                                            void *data)
{
  elf_image_cache_item_t *item = (elf_image_cache_item_t *)elf;
  setup_dwarf_state(&item->state, &elf->ei, error_cb, data);
  return &item->state;
}
